'****************************************************************************
'*                                                                          *
'*                  K   K  AAA  BBBB   OOO   OOO  M   M  !!!                *
'*                  K  K  A   A B   B O   O O   O MM MM  !!!                *
'*                  K K   A   A B   B O   O O   O M M M  !!!                *
'*                  KK    AAAAA BBBB  O   O O   O M M M  !!!                *
'*                  K K   A   A B   B O   O O   O M   M  !!!                *
'*                  K  K  A   A B   B O   O O   O M   M  !!!                *
'*                  K   K A   A BBBB   OOO   OOO  M   M   !                 *
'*                                                                          *
'* Kaboom! - A remake of the 1983 game by Activision for the Atari 2600     *
'* V1.0      (well, sort of)                                                *
'*                                                                          *
'* Terry Ritchie - terry.ritchie@gmail.com                                  *
'* September 26th, 2010                                                     *
'*                                                                          *
'* A convict is trying to escape from prison and somehow he got his hands   *
'* on an unlimited supply of bombs! It's up to you to catch the bombs in    *
'* the buckets of water you posess to extinguish them.                      *
'*                                                                          *
'* Use your mouse to move your buckets from right/left and catch the bombs. *
'*                                                                          *
'* This game is being released as freeware.                                 *
'*                                                                          *
'* This game was simply made to show novice programmers how a simple game   *
'* can be designed with QB64. All of the animation used in the game was     *
'* written using QB64's unique graphics handeling commands.                 *
'*                                                                          *
'* Future update plans:                                                     *
'*                                                                          *
'* - Use QB64's font commands to display score, menu options and high score *
'*   indication at end of game.                                             *
'* - Clean PNG graphics files to remove artifacts from around images.       *
'*                                                                          *
'****************************************************************************

'**
'** Declare functions and subroutines ***************************************
'**

DECLARE SUB UpdatePlayer() '                       update player movements
DECLARE SUB UpdatePrisoner() '                     update prisoner movements
DECLARE SUB UpdateBombs() '                        update bomb movements
DECLARE SUB TitleScreen() '                        main title screen
DECLARE SUB LevelUp(lvl AS INTEGER) '              increase level difficulty
DECLARE SUB Initialize() '                         initialize new game
DECLARE SUB CheckHighScore() '                     check/update high score
DECLARE FUNCTION FileExists (filename AS STRING) ' test for existance of file

'**
'** Declare CONSTants *******************************************************
'**

CONST False = 0, True = NOT False ' boolean truth testers
CONST Maxbombs = 40 '               maximum nuber of bombs on screen

'**
'** Declare TYPEs ***********************************************************
'**

TYPE bombs '               values for dropping bombs
    x AS INTEGER '         horizontal location of bomb (x)
    y AS INTEGER '         vertical location of bomb (y)
    speed AS INTEGER '     speed bomb is dropping
    image AS INTEGER '     which bomb image to show (fizzing fuse)
END TYPE

TYPE player '              values for player
    x AS INTEGER '         horizontal location of player (x)
    buckets AS INTEGER '   number of buckets player has
END TYPE

TYPE prisoner '            values for prisoner dropping bombs
    x AS INTEGER '         horizontal location of prisoner (x)
    xdir AS INTEGER '      direction of prisoner (right or left zig-zag)
    image AS INTEGER '     which prisoner image to show
END TYPE

'**
'** Declare variables *******************************************************
'**

DIM SHARED bomb(Maxbombs) AS bombs ' bombs that prisoner drops
DIM SHARED player AS player '        player properties
DIM SHARED prisoner AS prisoner '    prisoner properties
DIM SHARED dropFrequency% '          bomb drop frequency
DIM SHARED count% '                  general purpose counter
DIM SHARED hiScore% '                high score
DIM SHARED bombDrop% '               when to release a new bomb
DIM SHARED nextBomb% '               which bomb to release next
DIM SHARED level% '                  current game level
DIM SHARED shiftiness% '             how shifty is the prisoner
DIM SHARED kaboom% '                 true when bomb hits bottom
DIM SHARED levelClear% '             true when level cleared
DIM SHARED bombsToDrop% '            number of bombs to drop in level
DIM SHARED bombsDropped% '           number of bombs that have been dropped
DIM SHARED bombsCaught% '            number of bombs that have been caught
DIM SHARED score% '                  game score counter
DIM SHARED previousLevel% '          indicate when extra bucket awarded
DIM SHARED gameOver% '               true when game is over
DIM SHARED imgWall& '                game background image handler
DIM SHARED imgBucket& '              bucket image handler
DIM SHARED imgBomb&(1) '             bomb images handlers
DIM SHARED imgPrisoner&(1) '         prisoner images handlers
DIM SHARED imgTitle& '               game title screen image handler
DIM SHARED imgNewgame&(1) '          "start new game" images handlers
DIM SHARED imgQuit&(1) '             "quit" images handlers
DIM SHARED imgExplode& '             explosion image handler
DIM SHARED imgHiScore& '             message that displays high score
DIM SHARED sndExplode&(3) '          exploding bomb sounds
DIM SHARED sndHighScore& '           sound when high score is achieved
DIM SHARED sndLevelCleared& '        sound when level is cleared
DIM SHARED sndFuse& '                fuse burning sound
DIM SHARED sndSplash& '              splash sound when bomb hits water
DIM SHARED sndFizzle& '              sound of fuse burning out in water
DIM SHARED sndBackground& '          background music

'**
'** Initialize variables ****************************************************
'**

sndExplode&(1) = _SNDOPEN("explode1.wav", "VOL, SYNC") '   explosion1 sound
sndExplode&(2) = _SNDOPEN("explode2.wav", "VOL, SYNC") '   explosion2 sound
sndExplode&(3) = _SNDOPEN("explode3.wav", "VOL, SYNC") '   explosion3 sound
sndHighScore& = _SNDOPEN("highscore.wav", "VOL, SYNC") '   high score sound
sndLevelCleared& = _SNDOPEN("levelcleared.wav", "VOL, SYNC") ' level end snd
sndFuse& = _SNDOPEN("fuse.wav", "VOL, SYNC") '             fuse burn sound
sndSplash& = _SNDOPEN("splash.wav", "VOL, SYNC") '         splash sound
sndFizzle& = _SNDOPEN("fizzle.wav", "VOL, SYNC") '         fizzle out sound
imgWall& = _LOADIMAGE("wall.png", 32) '                    game background
imgBucket& = _LOADIMAGE("bucket.png", 32) '                bucket image
imgBomb&(0) = _LOADIMAGE("bomb1.png", 32) '                bomb 1 image
imgBomb&(1) = _LOADIMAGE("bomb2.png", 32) '                bomb 2 image
imgExplode& = _LOADIMAGE("explosion.png", 32) '            bomb explosion
imgPrisoner&(0) = _LOADIMAGE("prisoner.png", 32) '         prisoner1 (Bill)
imgPrisoner&(1) = _LOADIMAGE("prisoner2.png", 32) '        prisoner2 (Steve)
imgTitle& = _LOADIMAGE("title.png", 32) '                  title screen
imgNewgame&(0) = _LOADIMAGE("newgame.png", 32) '           new game image 1
imgNewgame&(1) = _LOADIMAGE("newgame2.png", 32) '          new game image 2
imgQuit&(0) = _LOADIMAGE("quit.png", 32) '                 quit image 1
imgQuit&(1) = _LOADIMAGE("quit2.png", 32) '                quit image 2
imgHiScore& = _LOADIMAGE("hiscore.png", 32) '              high score image

'****************************************************************************
'**
'** Main program code starts here
'**

SCREEN _NEWIMAGE(800, 600, 32) '             800 x 600 32bit color
_TITLE "Kaboom!" '                           set window title
RANDOMIZE TIMER '                            seed random number generator
DO '                                         begin main code loop
    Initialize '                             initialize game variables
    TitleScreen '                            display the title screen
    '**
    '** Select a random background song from 9 available
    '**
    sndBackground& = _SNDOPEN("kab" + LTRIM$(RTRIM$(STR$(INT(RND(1) * 9) + 1))) + ".mid", "VOL")
    '**
    _SNDVOL sndBackground&, .25 '            25% volume background music
    _SNDLOOP sndBackground& '                loop the background music
    DO '                                     begin game loop here
        LevelUp (level%) '                   set up the new level
        kaboom% = False '                    reset bomb hit indicator
        levelClear% = False '                reset level clear indicator
        bombsDropped% = 0 '                  reset number of bombs dropped
        bombsCaught% = 0 '                   reset number of bombs caught
        DO '                                 begin current level loop here
            _PUTIMAGE , imgWall& '           overwrite screen with background
            LOCATE 2, 50 '                   position text cursor
            '**
            '** print the current score to screen
            '**
            PRINT RIGHT$("0000" + LTRIM$(RTRIM$(STR$(score%))), 5);
            '**
            UpdatePlayer '                   update player bucket movements
            UpdatePrisoner '                 update the prisoner movements
            UpdateBombs '                    update the dropping bombs
            _LIMIT 64 '                      limit game 64 frames per second
            _DISPLAY '                       display changes made to screen
        LOOP UNTIL kaboom% OR levelClear% '  keep playing until boom or clear
        IF kaboom% THEN '                    did bomb hit ground?
            level% = level% - 1 '            send player back a level
            IF level% = 0 THEN level% = 1 '  too far?
            player.buckets = player.buckets - 1 ' take a player bucket away
            IF player.buckets < 1 THEN gameOver% = True ' all buckets gone?
        ELSE
            level% = level% + 1 '            level cleared, advance to next
            _SNDPLAY sndLevelCleared& '      play congrats chorus
        END IF
    LOOP UNTIL gameOver% '                   leave loop when game is over
    _SNDSTOP sndBackground& '                stop the background music
    _SNDCLOSE sndBackground& '               unload the background music
    CheckHighScore '                         New high score?
LOOP '                                       end of main code loop

'**
'** Main program code ends here
'**
'****************************************************************************

'**
'** Subroutines and Functions
'**

SUB CheckHighScore ()

'****************************************************************************
'*                                                                          *
'* Check current score against high score and update if needed              *
'*                                                                          *
'****************************************************************************

IF score% > hiScore% THEN '                                new high score?
    _AUTODISPLAY '                                         auto update screen
    _PUTIMAGE (0, 0), imgHiScore& '                        display message
    _DELAY 2 '                                             let it soak in
    _SNDPLAY sndHighScore& '                               play yahoo music
    _DELAY 5 '                                             time to celebrate
    OPEN "hiscore.kab" FOR OUTPUT AS #1 '                  open score file
    PRINT #1, score% '                                     write new score
    CLOSE #1 '                                             close score file
ELSE
    _DELAY 3 '                                             time to view score
END IF

END SUB

SUB Initialize ()

'****************************************************************************
'*                                                                          *
'* Initialize the variables for a new game                                  *
'*                                                                          *
'****************************************************************************

player.x = 400 '       start location of player bucket(s)
player.buckets = 3 '   number of buckets player starts with
prisoner.x = 400 '     start location of prisoner
prisoner.image = 0 '   start image of prisoner
prisoner.xdir = 0 '    start direction of prisoner
hiScore% = 0 '         set high score to 0
bombDrop% = 0 '        bomb drop timer
dropFrequency% = 64 '  how often to drop bombs (wait x fps, 64 = 1 sec)
nextBomb% = 0 '        bomb drop counter
level% = 1 '           start at level 1
shiftiness% = 1 '      how often prisoner changes direction randomly
kaboom% = False '      true when bomb hits ground
levelClear% = False '  true after each level cleared
score% = 0 '           reset game score
gameOver% = False '    reset game over indicator

FOR count% = 1 TO Maxbombs '  initialize all bombs
    bomb(count%).y = 0 '      start y position of each bomb
    bomb(count%).speed = 0 '  start drop speed of each bomb
    bomb(count%).image = 0 '  start image of each bomb (burning fuse anim)
NEXT count%

IF FileExists("hiscore.kab") THEN '       high score file exist?
    OPEN "hiscore.kab" FOR INPUT AS #1 '  yes, open it
    INPUT #1, hiScore% '                  get the high score
    CLOSE #1 '                            close high score file
END IF

END SUB

SUB LevelUp (lvl AS INTEGER)

'****************************************************************************
'*                                                                          *
'* Increases (or decreases) the level of difficulty                         *
'*                                                                          *
'****************************************************************************

DIM bombSpeed% ' speed at which bombs can drop

bombDrop% = 0 '                                            reset drop counter
SELECT CASE lvl '                                          which level?
    CASE 1 '                                               level 1
        previousLevel% = 1 '                               let next level know
        bombsToDrop% = 20 '                                drop 20 bombs
        bombSpeed% = 1 '                                   fall slowly
        dropFrequency% = 64 '                              1 bomb per sec
        prisoner.xdir = 1 '                                prisoner slow
        shiftiness% = 200 '                                zig-zag low
    CASE 2 '                                               level 2
        previousLevel% = 2 '                               let next level know
        bombsToDrop% = 30 '                                drop 30 bombs
        bombSpeed% = 2 '                                   fall a bit faster
        dropFrequency% = 60 '                              1.1 bombs per sec
        prisoner.xdir = 2 '                                prisoner bit faster
        shiftiness% = 180 '                                zig-zag faster
    CASE 3 '                                               level 3
        IF previousLevel% = 2 THEN previousLevel% = 3 '    let next level know
        bombsToDrop% = 35 '                                drop 35 bombs
        bombSpeed% = 2 '                                   fall same speed
        dropFrequency% = 44 '                              1.4 bombs per sec
        prisoner.xdir = 3 '                                prisoner faster
        shiftiness% = 160 '                                zig-zag faster
    CASE 4 '                                               level 4
        '**
        '** only award extra bucket if coming from previous level,
        '** not coming back from higher level (kaboom happened)
        '**
        IF previousLevel% = 3 THEN player.buckets = player.buckets + 1
        '**
        previousLevel% = 4 '                               let next level know
        bombsToDrop% = 40 '                                drop 40 bombs
        bombSpeed% = 3 '                                   fall even faster
        dropFrequency% = 40 '                              1.8 bombs per sec
        prisoner.xdir = 4 '                                prisoner very fast
        shiftiness% = 140 '                                zig-zag crazier
    CASE 5 '                                               level 5
        IF previousLevel% = 4 THEN previousLevel% = 5 '    let next level know
        bombsToDrop% = 45 '                                drop 45 bombs
        bombSpeed% = 3 '                                   fall same speed
        dropFrequency% = 32 '                              2 bombs per sec
        prisoner.xdir = 5 '                                prisoner faster!
        shiftiness% = 120 '                                zig-zag wild now
    CASE 6 '                                               level 6
        '**
        '** only award extra bucket if coming from previous level,
        '** not coming back from higher level (kaboom happened)
        '**
        IF previousLevel% = 5 THEN player.buckets = player.buckets + 1
        '**
        previousLevel% = 6 '                               let next level know
        bombsToDrop% = 55 '                                drop 55 bombs
        bombSpeed% = 4 '                                   fall even faster!
        dropFrequency% = 24 '                              2.4 bombs per sec
        prisoner.xdir = 6 '                                prisoner real fast
        shiftiness% = 100 '                                zig-zag irratic
    CASE ELSE '                                            awesome player!
        bombsToDrop% = 60 '                                drop 60 bombs
        bombSpeed% = 5 '                                   fall extreme fast
        dropFrequency% = 16 '                              3 bombs per sec
        prisoner.xdir = 7 '                                prisoner sprinting
        shiftiness% = 80 '                                 zig-zag crazy!
END SELECT
IF player.buckets > 3 THEN player.buckets = 3 '            only 3 buckets
RANDOMIZE TIMER '                                          seed rnd generator
FOR count% = 1 TO Maxbombs '                               set up bomb speeds
    bomb(count%).speed = INT(RND(1) * bombSpeed%) + 1 '    random speeds
NEXT count%

END SUB

SUB UpdateBombs ()

'****************************************************************************
'*                                                                          *
'* Updates the bombs falling on the screen                                  *
'*                                                                          *
'****************************************************************************

IF NOT _SNDPLAYING(sndFuse&) THEN '                        start fuse sound
    _SNDVOL sndFuse&, .25 '                                set volume low
    _SNDLOOP sndFuse& '                                    loop fuse sound
END IF
bombDrop% = bombDrop% + 1 '                                bomb timer
IF bombsDropped% <> bombsToDrop% THEN '                    more bombs to drop?
    IF bombDrop% = dropFrequency% THEN '                   do it now?
        bombsDropped% = bombsDropped% + 1 '                count dropped bombs
        bombDrop% = 0 '                                    reset bomb timer
        nextBomb% = nextBomb% + 1 '                        where in array?
        IF nextBomb% > Maxbombs THEN nextBomb% = 1 '       back to start array
        bomb(nextBomb%).x = prisoner.x - 25 '              position x location
        bomb(nextBomb%).y = 25 '                           position y location
        bomb(nextBomb%).image = 0 '                        fuse burn image
    END IF
END IF
FOR count% = 1 TO Maxbombs '                               cycle thru array
    IF bomb(count%).y <> 0 THEN '                          is bomb out there?
        bomb(count%).y = bomb(count%).y + bomb(count%).speed ' move bomb down
        bomb(count%).image = 1 - bomb(count%).image '      cycle fuse image
        '**
        '** draw the image of the bomb to the screen
        '**
        _PUTIMAGE (bomb(count%).x, bomb(count%).y), imgBomb&(bomb(count%).image)
        '**
        IF bomb(count%).y >= 550 THEN '                    bomb at bottom?
            kaboom% = True '                               yes, kaboom happens
            _SNDSTOP sndFuse& '                            stop fuse sound
        ELSE '                                             bomb still dropping
            splash% = False '                              no splash yet
            '**
            '** is bomb currently located where the bottom bucket resides?
            '** if so then add 1 to score and make splash true
            '**
            IF bomb(count%).x >= player.x - 50 AND bomb(count%).x <= player.x + 25 AND bomb(count%).y > 500 AND bomb(count%).y < 560 THEN
                splash% = True
                score% = score% + 1
            END IF
            '**
            '** is bomb currently located where the middle bucket resides?
            '** if so does the middle bucket exist?
            '** if so then add 2 to score and make splash true
            '**
            IF player.buckets > 1 AND bomb(count%).x >= player.x - 50 AND bomb(count%).x <= player.x + 25 AND bomb(count%).y > 400 AND bomb(count%).y < 460 THEN
                splash% = True
                score% = score% + 2
            END IF
            '**
            '** is bomb currently located where the top bucket resides?
            '** if so does the top bucket exist?
            '** if so then add 3 to score and make splash true
            '**
            IF player.buckets > 2 AND bomb(count%).x >= player.x - 50 AND bomb(count%).x <= player.x + 25 AND bomb(count%).y > 300 AND bomb(count%).y < 360 THEN
                splash% = True
                score% = score% + 3
            END IF
            IF splash% THEN '                              bomb hit bucket?
                bombsCaught% = bombsCaught% + 1 '          count caught bombs
                IF bombsCaught% = bombsToDrop% THEN '      all caught?
                    levelClear% = True '                   level is clear
                    _SNDSTOP sndFuse& '                    stop fuse sound
                END IF
                bomb(count%).y = 0 '                       remove bomb
                _SNDPLAY sndSplash& '                      make splash sound
                _SNDPLAY sndFizzle& '                      fuse fizzles out
            END IF
        END IF
    END IF
NEXT count%
IF kaboom% THEN '                                          bomb hit ground?
    _PUTIMAGE (prisoner.x - 25, 0), imgPrisoner&(1) '      Ballmer's head!
    FOR count% = 1 TO Maxbombs '                           cycle through array
        IF bomb(count%).y <> 0 THEN '                      bomb in play?
            '**
            '** place an explosion image over bomb on screen
            '**
            _PUTIMAGE (bomb(count%).x - 10, bomb(count%).y + 10), imgExplode&
            '**
            bomb(count%).y = 0 '                           remove bomb
            _SNDPLAY sndExplode&(INT(RND(1) * 3) + 1) '    which boom sound?
            _DELAY .2 '                                    pause between booms
            _DISPLAY '                                     update screen
        END IF
    NEXT count%
END IF

END SUB

SUB UpdatePlayer ()

'****************************************************************************
'*                                                                          *
'* Updates the position of the player's bucket(s)                           *
'*                                                                          *
'****************************************************************************

DO WHILE _MOUSEINPUT '                                     get mouse changes
LOOP '                                                     since last visit
player.x = INT(_MOUSEX / 1.14286) + 50 '                   calc bucket posit
FOR count% = 0 TO player.buckets - 1 '                     draw each bucket
    buckety% = 520 - (count% * 100) '                      y bucket location
    _PUTIMAGE (player.x - 50, buckety%), imgBucket& '      display bucket
NEXT count%

END SUB

SUB UpdatePrisoner ()
'****************************************************************************
'*                                                                          *
'* Updates the position of the prisoner                                     *
'*                                                                          *
'****************************************************************************

prisoner.x = prisoner.x + prisoner.xdir '                  move prisoner
IF prisoner.x > 775 THEN '                                 hit right edge
    prisoner.x = 775 '                                     stop prisoner
    prisoner.xdir = -prisoner.xdir '                       reverse direction
END IF
IF prisoner.x < 25 THEN '                                  hit left edge
    prisoner.x = 25 '                                      stop prisoner
    prisoner.xdir = -prisoner.xdir '                       reverse direction
END IF
'**
'** based on shiftiness should we randonly shift prisoner?
'**
IF INT(RND(1) * shiftiness%) = INT(shiftiness% / 2) THEN prisoner.xdir = -prisoner.xdir
'**
_PUTIMAGE (prisoner.x - 25, 0), imgPrisoner&(0) '          display prisoner

END SUB

SUB TitleScreen ()
'****************************************************************************
'*                                                                          *
'* Main title screen                                                        *
'*                                                                          *
'****************************************************************************

DIM sndIntroMusic& ' intro music handler
DIM ng% '            controls which "start new game" image to display
DIM qt% '            controls which "quit" image to display
DIM optionChosen% '  holds the option player chooses
DIM mx% '            x location of mouse pointer
DIM my% '            y location of mouse pointer
DIM mb% '            holds status of left mouse button
DIM sndMarvin& '     marvin martian "where's the kaboom" sound handler
DIM highScore$ '     high score scrolling message to display on screen

ng% = 0 '            show unhighlighted new game option first
qt% = 0 '            show unhighlighted quit option first
optionChosen% = 0 '  use has not yet chosen an option

_FULLSCREEN '                                              go full screen
_MOUSESHOW '                                               show mouse pointer
_AUTODISPLAY '                                             auto update screen
sndIntroMusic& = _SNDOPEN("title.ogg", "VOL") '            load title music
_SNDVOL sndIntroMusic&, .5 '                               set title volume
_SNDLOOP sndIntroMusic& '                                  loop title music
_PUTIMAGE , imgTitle& '                                    display title image
'**
'** form the scrolling high score message
'**
highScore$ = STRING$(40, 32) + "The score to beat is "
highScore$ = highScore$ + RIGHT$("0000" + LTRIM$(RTRIM$(STR$(hiScore%))), 5)
highScore$ = highScore$ + STRING$(40, 32) + "Can you keep Billy boy in jail? "
'**
COLOR _RGB(0, 0, 0), _RGB(85, 183, 255) '                  set text color
count% = 0 '                                               text scroll count
DO '                                                       start input loop
    _LIMIT 8 '                                             don't hog cpu
    count% = count% + 1 '                                  inc scroll count
    IF count% > LEN(highScore$) THEN count% = 1 '          reset count?
    LOCATE 2, 40 '                                         locate text
    PRINT MID$(highScore$, count%, 20); '                  print banner
    DO WHILE _MOUSEINPUT '                                 get mouse changes
    LOOP '                                                 since last visit
    mx% = _MOUSEX '                                        x location of mouse
    my% = _MOUSEY '                                        y location of mouse
    mb% = _MOUSEBUTTON(1) '                                left button status
    '**
    '** if the mouse pointer falls within either square containing a menu
    '** option then set the appropriate variable controlling which image
    '** to show to 1, otherwise set each one to 0 (the default image)
    '**
    IF mx% > 49 AND mx% < 360 AND my% > 274 AND my% < 301 THEN ng% = 1 ELSE ng% = 0
    IF mx% > 149 AND mx% < 241 AND my% > 339 AND my% < 373 THEN qt% = 1 ELSE qt% = 0
    '**
    _PUTIMAGE (50, 275), imgNewgame&(ng%) '                display menu entry
    _PUTIMAGE (150, 340), imgQuit&(qt%) '                  display menu entry
    IF ng% = 1 AND mb% <> 0 THEN optionChosen% = 1 '       option 1 chosen
    IF qt% = 1 AND mb% <> 0 THEN optionChosen% = 2 '       option 2 chosen
LOOP UNTIL optionChosen% <> 0 '                            end of input loop
_SNDSTOP sndIntroMusic& '                                  stop title music
_SNDCLOSE sndIntroMusic& '                                 close title music
_MOUSEHIDE '                                               hide mouse pointer
IF optionChosen% = 2 THEN '                                did user quit?
    LOCATE 2, 40 '                                         locate text
    PRINT "Bill said "; CHR$(34); "Chicken!"; CHR$(34); '  taunt player!
    sndMarvin& = _SNDOPEN("marvin.wav", "VOL") '           load marvin sound
    _SNDPLAY sndMarvin& '                                  play marvin sound
    _DELAY 5 '                                             wait 5 seconds
    _SNDCLOSE sndMarvin& '                                 close marvin sound
    COLOR _RGB(255, 255, 255), _RGB(0, 0, 0) '             reset text color
    SYSTEM '                                               return to OS
END IF

END SUB

FUNCTION FileExists (filename AS STRING)

'****************************************************************************
'*                                                                          *
'* Tests for the existance of a file. Returns true if file found, false     *
'* otherwise.                                                               *
'*                                                                          *
'****************************************************************************

OPEN filename FOR APPEND AS #1 '                           open file
IF LOF(1) THEN '                                           data present?
    FileExists = True '                                    file does exist
ELSE
    FileExists = False '                                   file not exist
END IF
CLOSE #1 '                                                 close file

END FUNCTION

